import { useEffect, useCallback } from "react";
import { motion } from "framer-motion";
import {
  Button,
  Box,
  Alert,
  AlertIcon,
  AlertTitle,
  AlertDescription,
} from "@chakra-ui/react";
import { useQuery } from "@tanstack/react-query";
import { BeatLoader } from "react-spinners";
import { SvgArrow } from "@/app/_components/Svg";
import {
  useMenuContext,
  useDispatchContext,
  useDrawerContext,
} from "@/app/_context";

import { isObjectEmpty } from "@/app/_utils";

import {
  planyourTripMenuKeys,
  menuData
} from "@/app/_includes/Header/components/_data";

import AppDrawerLink from "./AppDrawerLink";

// Api Loader
const ApiLoader = () => {
  return (
    <Box pt={40} display="flex" justifyContent="center" alignItems="center">
      <BeatLoader
        loading={true}
        size={20}
        color="#FF9D00"
        cssOverride={{
          display: "block",
          margin: "0 auto",
        }}
      />
    </Box>
  );
};

const ApiError = ({ message }) => {
  return (
    <Alert status="error">
      <AlertIcon />
      <AlertTitle>API ERROR!.</AlertTitle>
      <AlertDescription>{message}</AlertDescription>
    </Alert>
  );
};

export default function MenuChildren() {
  const dispatch = useDispatchContext();
  const { data } = useMenuContext();
  const {
    activeKey,
    menuList,
    selectedId,
    prevId,
    slug,
    learnMore,
    activeDepth,
    depth,
  } = useDrawerContext();

  const {
    data: subMenu,
    isInitialLoading,
    isError,
    error,
  } = useQuery({
    queryKey: ["submenu", slug],
    retry: false,
    queryFn: () => fetch(`/api/submenu/${slug}`).then((res) => res.json()),
    enabled: !!slug,
  });

  // Handle Effect for watching submenu from
  // Api
  useEffect(() => {
    if (subMenu) {
      const {
        details: { items, learn_more },
      } = subMenu;
      dispatch({
        type: "SET_MENU_LIST",
        value: [],
      });
      dispatch({
        type: "SET_MENU_LIST",
        value: items,
      });
      dispatch({
        type: "SET_LEARN_MORE",
        value: learn_more,
      });
    }
  }, [subMenu]);

  // Effect for checking the Menu And Learn More Items
  // Based on the ActiveKey, SelectedId and PrevId
  useEffect(() => {
    if (selectedId && activeKey) {
      if (depth === 3 && activeDepth + 1 === depth) {
        dispatch({
          type: "SET_SLUG",
          value: menuList.find((item) => item?._id === selectedId)?.link.slug,
        });
        dispatch({ type: "SET_ACTIVE_DEPTH", value: activeDepth + 1 });
      } else {
        dispatch({
          type: "SET_MENU_LIST",
          value:
            menuList.find((item) => item._id === selectedId)?.subMenu ?? [],
        });
        dispatch({
          type: "SET_LEARN_MORE",
          value:
            menuList.find((item) => item._id === selectedId)?.learn_more ?? {},
        });
        dispatch({ type: "SET_ACTIVE_DEPTH", value: activeDepth + 1 });
      }
    } else {
      if (prevId) {
        const subMenu = data[activeKey]["menu"].find(
          (item) => item._id === prevId
        )?.subMenu;
        dispatch({
          type: "SET_MENU_LIST",
          value: subMenu,
        });
        dispatch({
          type: "SET_SLUG",
          value: "",
        });
      } else {
        if (activeKey === "plan_your_trip")
          dispatch({
            type: "SET_MENU_LIST",
            value: planyourTripMenuKeys.map((item) => data[activeKey]["menu"][item]),
          });
        else
          dispatch({ type: "SET_MENU_LIST", value: data[activeKey]["menu"] });
      }
    }
  }, [activeKey, selectedId, prevId]);

  // Fun for setting the id of the Selected Menu Item
  const handleSetId = (id) => {
    if (selectedId && !slug) {
      dispatch({ type: "SET_PREVIOUS_ID", value: selectedId });
    }
    dispatch({ type: "SET_SELECTED_ID", value: id });
  };

  // Handling Back func for the Sub Menu List
  const handleBack = () => {
    if (selectedId) dispatch({ type: "SET_SELECTED_ID", value: "" });

    if (prevId && !slug) dispatch({ type: "SET_PREVIOUS_ID", value: "" });

    dispatch({ type: "SET_LEARN_MORE", value: {} });
    dispatch({ type: "SET_ACTIVE_DEPTH", value: activeDepth - 1 });
  };

  // Loading Component
  if (isInitialLoading) return <ApiLoader />;

  // Api Error Component
  if (isError) return <ApiError message={error.message} />;

  if (menuList.length > 0) {
    return (
      <>
        <Box m={6}>
          <Button
            variant="outline"
            height={8}
            as={motion.button}
            size="md"
            width="100px"
            borderColor="black"
            borderRadius="full"
            whileHover="hover"
            onClick={() => handleBack()}
            initial={false}
            _hover={{
              "svg path": {
                fill: "orangePeel.900",
              },
            }}
          >
            <SvgArrow direction="rtl" color="black" />
            &nbsp; Back
          </Button>
        </Box>

        {menuList.map((item, idx) => {
          if (depth === activeDepth) {
            let ur=(item?.link?.parent ? `/${item.link.parent}` : "").concat(item?.link?.slug ? `/${item.link.slug}` : "")
            return (
              <AppDrawerLink
                key={idx}
                menuKey={item?.title ?? item?.label}
                title={item?.title ?? item?.label}
                text={item?.title ?? item.label}
                // href={`/${item?.link?.parent?.concat(
                //   item?.link?.slug ? `/${item.link.slug}` : ""
                // )}`}
                href={activeKey === "plan_your_trip" 
                ? item.link 
                : ur
              }
                type={activeKey === "plan_your_trip" 
                ? item.type 
                : item.link.type }
              />
            );
          } else {
            return (
              <Button
                p={6}
                borderBottom="1px"
                borderBottomColor="blackAlpha.400"
                borderRadius="none"
                variant="solid"
                display="flex"
                alignItems="center"
                justifyContent="flex-start"
                key={idx}
                onClick={() => handleSetId(item._id)}
              >
                {item.label}
              </Button>
            );
          }
        })}
        {learnMore && !isObjectEmpty(learnMore) && (
          <AppDrawerLink
            title={learnMore.link.title}
            text={learnMore.link.text}
            href={`/${learnMore.link.parent.concat(
              learnMore.link.slug ? `/${learnMore.link.slug}` : ""
            )}`}
            menuKey={learnMore.link.title}
            type={learnMore.link.type}
          />
        )}
      </>
    );
  }

  return null;
}
