"use client";

import { bool, node } from "prop-types";
import { createContext, useContext, useReducer } from "react";
import { useQuery } from "@tanstack/react-query";

import {
  authReducer,
  drawerReducer,
  lightBoxReducer,
  menuReducer,
} from "./_reducers";

/********************  Common Dispatch Context Hook  ************************/

// Common Dispatch Context
export const DispatchContext = createContext(() => {});

DispatchContext.displayName = "KTDispatcher";

// Custom hooks for dispatch context
const useDispatchContext = () => {
  const value = useContext(DispatchContext);

  if (value === null) throw new Error("Dispatch Provider is missing");

  return value;
};

/********************  Common Dispatch Context Hook  ************************/

/********************  Device Context Hook And Provider  ************************/

// Device Context
const DeviceContext = createContext(null);

DeviceContext.displayName = "DeviceContextProvider";

// CustomHooks for Device Context
const useDeviceContext = () => {
  const value = useContext(DeviceContext);

  if (value === null) throw new Error("Device Provider is Missing");

  return value;
};

// Device Context Provider
const DeviceContextProvider = ({ children, isMobile, isDesktop, isTablet }) => {
  return (
    <DeviceContext.Provider
      value={{
        isDesktop,
        isTablet,
        isMobile,
      }}
    >
      {children}
    </DeviceContext.Provider>
  );
};

DeviceContextProvider.propTypes = {
  children: node.isRequired,
  isDesktop: bool,
  isMobile: bool,
  isTablet: bool,
};

/********************  Device Context Hook And Provider  ************************/

/********************  Menu Context Hook And Provider  ************************/

// Mega Menu Context
export const MenuContext = createContext(null);

MenuContext.displayName = "KTMenuProvider";

// Custom Hooks For Mega Menu Context
const useMenuContext = () => {
  const value = useContext(MenuContext);

  if (value === null) throw new Error("MegaMenu Provider is Missing");

  return value;
};

// Menu Context Provider
const MenuContextProvider = ({ children, data }) => {
  const [state, dispatch] = useReducer(menuReducer, {
    data,
    isOpen: false,
    hasChildren: false,
    isOpenSearch: false,
    activeKey: "",
  });

  return (
    <DispatchContext.Provider value={dispatch}>
      <MenuContext.Provider
        value={{
          data: state.data,
          isOpen: state.isOpen,
          activeKey: state.activeKey,
          isOpenSearch: state.isOpenSearch,
          hasChildren: state.hasChildren,
        }}
      >
        {children}
      </MenuContext.Provider>
    </DispatchContext.Provider>
  );
};

MenuContextProvider.propTypes = {
  children: node.isRequired,
};

/********************  Menu Context Hook And Provider ************************/

/******************** Sidebar Drawer Context Hook And Provider ***************/

// Sidebar Drawer Context
const DrawerContext = createContext(null);

DrawerContext.displayName = "KTDrawerProvider";

// Drawer context custom hook
const useDrawerContext = () => {
  const value = useContext(DrawerContext);

  if (value === null) throw new Error("Drawer Provider is Missing");

  return value;
};

// Drawer Context Provider
const DrawerContextProvider = ({ children }) => {
  const [state, dispatch] = useReducer(drawerReducer, {
    activeKey: "",
    depth: 0,
    activeDepth: 0,
    menuList: [],
    learnMore: {},
    selectedId: "",
    prevId: "",
    slug: "",
  });

  return (
    <DispatchContext.Provider value={dispatch}>
      <DrawerContext.Provider
        value={{
          activeKey: state.activeKey,
          depth: state.depth,
          activeDepth: state.activeDepth,
          menuList: state.menuList,
          learnMore: state.learnMore,
          selectedId: state.selectedId,
          prevId: state.prevId,
          slug: state.slug,
        }}
      >
        {children}
      </DrawerContext.Provider>
    </DispatchContext.Provider>
  );
};

DrawerContextProvider.propTypes = {
  children: node.isRequired,
};

/******************** Sidebar Drawer Context Hook And Provider ***************/

/******************** Auth Context Hook And Provider ***************/

// Auth Context
const AuthContext = createContext(null);

AuthContext.displayName = "KTAuthProvider";

// Auth Context hook
const useAuthContext = () => {
  const value = useContext(AuthContext);

  if (value === null) throw new Error("Auth Provider is Missing");

  return value;
};

// Auth Provider
const AuthContextProvider = ({ children }) => {
  const [state, dispatch] = useReducer(authReducer, {
    isOpenSignUp: false,
    isOpenSignIn: false,
  });

  return (
    <DispatchContext.Provider value={dispatch}>
      <AuthContext.Provider
        value={{
          isOpenSignUp: state.isOpenSignUp,
          isOpenSignIn: state.isOpenSignIn,
        }}
      >
        {children}
      </AuthContext.Provider>
    </DispatchContext.Provider>
  );
};

AuthContextProvider.propTypes = {
  children: node.isRequired,
};

/******************** Auth Context Hook And Provider ***************/

/*************** Gallery Light Box Context Hook And Provider *******/

// lightbox Context
const LightBoxContext = createContext(null);

LightBoxContext.displayName = "KTLightBoxProvider";

// useLightbox Context Hook
const useLightboxContext = () => {
  const value = useContext(LightBoxContext);

  if (value === null) throw new Error("Light Box Provider Is Missing");

  return value;
};

// Light Box Provider
const LightBoxContextProvider = ({ children, data }) => {
  const { gallery } = data;

  // Loop Through the gallery data and add
  // New Information Such As Idx To identify the
  // Uniqueness
  Object.keys(gallery).forEach((item) => {
    let idx = 0;
    for (const i of gallery[item]) {
      gallery[item][idx] = {
        idx,
        ...i,
      };
      idx++;
    }
  });

  const [state, dispatch] = useReducer(lightBoxReducer, {
    gallery,
    navKeys: Object.keys(gallery),
    activeKey: Object.keys(gallery).at(0),
    activeIndex: 0,
    direction: 1,
  });

  return (
    <DispatchContext.Provider value={dispatch}>
      <LightBoxContext.Provider
        value={{
          gallery: state.gallery,
          navKeys: state.navKeys,
          activeKey: state.activeKey,
          activeIndex: state.activeIndex,
          direction: state.direction,
        }}
      >
        {children}
      </LightBoxContext.Provider>
    </DispatchContext.Provider>
  );
};

LightBoxContextProvider.propTypes = {
  children: node.isRequired,
};

/*************** Gallery Light Box Context Hook And Provider *******/

/************** Application Configuration Provider ****************/

const ConfigurationContext = createContext(null);

ConfigurationContext.displayName = "KTConfiguration";

// useConfiguration Context Hook
const useConfigurationContext = () => {
  const value = useContext(ConfigurationContext);

  if (value === null) throw new Error("Configuration Provider Is Missing");

  return value;
};

// Configuration Provider

const ConfigurationProvider = ({ children }) => {
  const { data } = useQuery({
    queryKey: ["configuration"],
    queryFn: () => fetch(`/api/configuration`).then((res) => res.json()),
    retry: false,
  });

  return (
    <ConfigurationContext.Provider value={{ data }}>
      {children}
    </ConfigurationContext.Provider>
  );
};

ConfigurationProvider.propTypes = {
  children: node.isRequired,
};

/************** Application Configuration Provider ****************/

export {
  AuthContextProvider,
  DeviceContextProvider,
  MenuContextProvider,
  DrawerContextProvider,
  LightBoxContextProvider,
  ConfigurationProvider,
  useAuthContext,
  useMenuContext,
  useDeviceContext,
  useDrawerContext,
  useLightboxContext,
  useConfigurationContext,
  useDispatchContext,
};
