"use client";

import { useRef, useState, useEffect, useCallback } from "react";

import { Link } from "react-scroll";
import { Container, Box, List, ListItem, Divider } from "@chakra-ui/react";
import { appTextFontSize } from "../../_styles";

export default function ScrollHeading({ items, has_sub_banner }) {
  const [isInView, setIsInView] = useState(false);
  const [offsetValue, setOffsetValue] = useState(0);

  const WrapperElem = has_sub_banner ? Container : Box;

  const stickyRef = useRef();
  const scrollRef = useRef(null);
  const scrolledElemRefs = useRef(null);

  let boxProps = {
    ...(isInView && {
      pos: "fixed",
      top: 0,
      left: 0,
      right: 0,
      bg: "white",
      transition: "0.5s linear",
      zIndex: "sticky",
    }),
    ...(has_sub_banner && {
      maxW: "container.2xl",
    }),
    ...(has_sub_banner &&
      isInView && {
        maxW: "100%",
        px: 0,
      }),
  };

  // getting ref map for scroll elements
  const getRefMap = () => {
    if (!scrolledElemRefs.current) {
      scrolledElemRefs.current = new Map();
    }
    return scrolledElemRefs.current;
  };

  // Handler for when scroll menu key is active
  const handleElemActive = (itemKey) => {
    if (scrollRef?.current?.scrollWidth > document.documentElement.offsetWidth) {
      const map = getRefMap();
      const node = map.get(itemKey);
      if (node) {
        const { x } = node.getBoundingClientRect();
        scrollRef.current.scrollLeft += x;
      }      
    }
  };

  // Handler for window scroll
  const onScroll = useCallback(() => {
    const sticky = stickyRef.current.offsetTop;
    if (sticky > 0) setOffsetValue(sticky);
    if (window.scrollY > offsetValue && offsetValue > 0) setIsInView(true);
    else setIsInView(false);
  }, [offsetValue, setIsInView]);

  // Effect for registering the window scroll
  // With dependency onScroll fn
  useEffect(() => {
    window.addEventListener("scroll", onScroll);
    return () => {
      window.removeEventListener("scroll", onScroll);
    };
  }, [onScroll]);

  return (
    <WrapperElem ref={stickyRef} {...boxProps}>
      <List
        ref={scrollRef}
        display="flex"
        gap={5}
        /* justifyContent={{
          base: "flex-start",
          ...(items.length <= 6 && {
            md: "center",
          }),
          ...(items.length <= 6 && {
            lg: "center",
          }),
          xl: "center",
        }} */
        justifyContent="safe center"
        px={5}
        py={5}
        whiteSpace="nowrap"
        overflowX="auto"
        scrollBehavior="smooth"
        sx={{
          "li a.active": {
            color: "orangePeel.900",
          },
        }}
      >
        {items.map(({ _id, heading, key }) => (
          <ListItem
            key={_id}
            ref={(node) => {
              const map = getRefMap();
              if (node) {
                map.set(key, node);
              } else {
                map.delete(key);
              }
            }}
            cursor="pointer"
            _hover={{
              color: "orangePeel.900",
            }}
            _notFirst={{
              _before: {
                content: '"|"',
                marginRight: "5",
                color: "black",
              },
            }}
            sx={{
              "a": {
                fontSize: appTextFontSize
              }
            }}
          >
            <Link
              key={_id}
              to={key}
              spy={true}
              smooth={true}
              duration={500}
              onSetActive={(to) => handleElemActive(to)}
              offset={-80}
            >
              {heading.toUpperCase()}
            </Link>
          </ListItem>
        ))}
      </List>
      <Divider borderColor="#707070" borderBottomWidth="1px" opacity={0.25} />
    </WrapperElem>
  );
}
