import { useEffect, useState, useCallback, useRef } from "react";
import { motion, AnimatePresence, MotionConfig } from "framer-motion";
import { Box, Flex } from "@chakra-ui/react";

import VideoPlayer from "./_components/VideoPlayer";
import SliderInfoWrapper from "./_components/SliderInfoWrapper";
import SliderDescription from "./_components/SliderDescription";
import SliderNavigation from "./_components/SliderNavigation";
import VideoControls from "./_components/VideoControls";

import useMobile from "@/app/_hooks/useMobile";
import useNoOfSlider from "../../_hooks/useNumberOfSlider";

import { sliderSectionStyles } from "./_styles";

export default function Main({ items: bannerInfo }) {

  const [pageLoaded, setPageLoaded] = useState(false);

  useEffect(() => {
    setPageLoaded(true);
  }, []);
  // config to set how much slider need to show in the Main Navigation
  const [noOfSlider] = useNoOfSlider();
  const [isMobile] = useMobile();

  // Important states for the slider Navigation
  const [sliders, setSliders] = useState(() => bannerInfo);
  const [activeSliderId, setActiveSliderId] = useState(() => sliders[0]._id);
  const [sliderElemPos, setSliderElemPos] = useState(1);
  const [activeSlider, setActiveSlider] = useState(() => bannerInfo[0]);
  const [direction, setDirection] = useState(1);

  const {
    video: { desktop, mobile },
  } = activeSlider;

  // States for video element to track the play percentage
  // audio and isplaying
  const [playPercentage, setPlayPercentage] = useState(0);
  const [audioState, setAudioState] = useState(true);
  const [playing, setPlaying] = useState(true);

  // Video element ref
  const videoRef = useRef(null);

  // Effect for adding Active Slider Id
  useEffect(() => {
    setPlayPercentage(0);
    setActiveSliderId(sliders[sliderElemPos - 1]._id);
    setActiveSlider(sliders[sliderElemPos - 1]);

    if (videoRef?.current) {
      handleVideoStates("play");
    }
  }, [sliders, sliderElemPos, videoRef]);

  /**
   * Handling Video States / Play / Pause / Audio
   */
  const handleVideoStates = useCallback(
    (type) => {
      if (type === "play") {
        videoRef.current.play();
        setPlaying(true);
      } else if (type === "pause") {
        videoRef.current.pause();
        setPlaying(false);
      } else if (type === "audio") {
        setAudioState(!audioState);
      }
    },
    [audioState]
  );

  /**
   * Handle Next of the Slider
   */
  const handleNext = useCallback(() => {
    if (sliderElemPos === noOfSlider) {
      const sliderClone = [...sliders];
      const shiftedElem = sliderClone.shift();
      setSliders([...sliderClone, shiftedElem]);
    } else {
      setSliderElemPos((prevState) => (prevState += 1));
    }
    setDirection(1);
  }, [sliderElemPos, noOfSlider, sliders]);

  /**
   * Handling the element position of the slider
   * @param {int} elementPosition
   *
   */
  const handleSelectSlider = useCallback((elemPos) => {
    setSliderElemPos(elemPos + 1);
  }, []);

  /**
   * Handling prev state of the slider
   */
  const handlePrev = useCallback(() => {
    if (sliderElemPos <= 1) {
      const sliderClone = [...sliders];
      const poppedElem = sliderClone.pop();
      setSliders([poppedElem, ...sliderClone]);
    } else {
      setSliderElemPos((prevState) => (prevState -= 1));
    }
    setDirection(0);
  }, [sliders, sliderElemPos]);

  /**
   * Updating the progress of the video
   */
  const handleProgressUpdate = useCallback(() => {
    const duration = videoRef.current?.duration;
    const elapsedTime = videoRef.current?.currentTime;

    if (typeof duration !== "undefined" && typeof elapsedTime !== "undefined") {
      setPlayPercentage(Math.floor((elapsedTime / duration) * 100));
    }
  }, []);

  return (
    <MotionConfig
      transition={{
        x: { type: "spring", stiffness: 300, damping: 30, duration: 0.6 },
        opacity: { duration: 0.6 },
      }}
    >
      <Box as={motion.section} {...sliderSectionStyles}>
        <VideoPlayer
          ref={videoRef}
          activeSliderId={activeSliderId}
          onTimeUpdate={handleProgressUpdate}
          onEnded={handleNext}
          audioState={audioState}
          {...desktop}          
          url={pageLoaded ? desktop.url : null}
        />
        <SliderInfoWrapper>
          <AnimatePresence mode="wait">
            <SliderDescription {...activeSlider} key={activeSlider._id} />
          </AnimatePresence>
          {noOfSlider > 0 && (
            <Flex align="center" justify="space-between">
              <SliderNavigation
                sliders={sliders}
                noOfSlider={noOfSlider}
                activeSliderId={activeSlider._id}
                playPercentage={playPercentage}
                handleSelectSlider={handleSelectSlider}
                handleNext={handleNext}
                handlePrev={handlePrev}
                direction={direction}
                setDirection={setDirection}
              />
              <VideoControls
                playing={playing}
                audioState={audioState}
                handleVideoStates={handleVideoStates}
              />
            </Flex>
          )}
        </SliderInfoWrapper>
      </Box>
    </MotionConfig>
  );
}
