import dynamic from "next/dynamic";

import {
  Card,
  Image,
  Text,
  Flex,
  Box,
  Button,
  useDisclosure,
} from "@chakra-ui/react";
import { motion } from "framer-motion";
import { AppLink } from "@/app/_components";

import { SvgArrow } from "@/app/_components/Svg";
import { isObjectEmpty } from "@/app/_utils";

const AppModal = dynamic(() =>
  import("@/app/_components").then((mod) => mod.AppModal)
);

const ViewMoreButton = ({ onOpen }) => {
  return (
    <Button
      variant="outline"
      as={motion.button}
      size="md"
      width="100px"
      borderColor="black"
      borderRadius="full"
      whileHover="hover"
      onClick={() => onOpen()}
      initial={false}
      _hover={{
        "svg path": {
          fill: "orangePeel.900",
        },
      }}
    >
      View&nbsp;
      <SvgArrow color="black" />
    </Button>
  );
};

export default function NewsListItem({
  type,
  heading,
  description,
  media_icon_url = "",
  published_at = "",
  link = {},
  content = [],
}) {
  const { isOpen, onOpen, onClose } = useDisclosure();

  return (
    <>
      <Card px={6} py={6} h="full" borderRadius="lg">
        <Flex direction="column" gap={6}>
          {type === "award" && (
            <Image src="/icons/award.svg" width={45} h={45} alt={heading} />
          )}

          {type === "news" && media_icon_url && (
            <Image src={media_icon_url} width={240} h={33} alt={"Media"} />
          )}

          {heading && (
            <Text as="h5" fontSize="26" fontWeight="semibold">
              {heading}
            </Text>
          )}

          {published_at && (
            <Text fontWeight="semibold" color="orangePeel.900">
              {published_at}
            </Text>
          )}

          <Text>{description}</Text>

          {content.length > 0 && <ViewMoreButton onOpen={onOpen} />}

          {!isObjectEmpty(link) && (
            <Box display="inline-flex">
              <AppLink
                type={link.type}
                parent={link.parent}
                slug={link.slug}
                title={link.title}
                variant="pill"
                color="black"
                arrow
              >
                {link.text}
              </AppLink>
            </Box>
          )}
        </Flex>
      </Card>
      {content.length > 0 && (
        <AppModal
          size="xl"
          closeBtnProps={{
            zIndex: 99,
            pos: "absolute",
            top: "-10px",
            right: "-10px",
            color: "white",
            bg: "orangePeel.900",
            borderRadius: "full",
          }}
          isOpen={isOpen}
          onClose={onClose}
        >
          <Flex direction="column" gap={6} pt={6} pb={6}>
            <Text as="h5" fontSize="26" fontWeight="semibold">
              {heading}
            </Text>
            {content.map((item, idx) => (
              <Text key={idx}>{item}</Text>
            ))}
          </Flex>
        </AppModal>
      )}
    </>
  );
}
