import { Box, Flex, Text, IconButton } from "@chakra-ui/react";
import { BsGlobe, BsTelephone, BsFillEnvelopeFill } from "react-icons/bs";
import Link from 'next/link';

import {
  Card,
  AppLink,
  AppLocationButton,
  KtHeading,
  AppImage,
} from "@/app/_components";

import { SvgArrow } from "@/app/_components/Svg";
import { isObjectEmpty } from "@/app/_utils";

export default function InfoCardListItem({
  heading,
  image = {},
  embedcode,
  description,
  contact_no,
  address,
  district,
  state,
  address_main = "",
  email,
  website,
  location_url,
  cities = [],
  external_links = [],
  helpline_numbers = {},
  link,
}) {
  let prnt, trg, ur;
  if (link) {
    if (link.type === "internal" || link.type === "link") {
      prnt = process.env.NEXT_PUBLIC_APP_BASE_URL + link.parent;
    } else if (link.type === "internal-other") {
      prnt = process.env.NEXT_PUBLIC_OLD_URL + link.parent;
    } else {
      prnt = link.parent;
    }
    ur = (prnt ?? "").concat(link.slug ?? "");
    trg = (link.type !== "internal-other" && link.type !== "external") ? "" : "_blank";
  }

  const headingElement = <KtHeading as="h6">{heading}</KtHeading>;
  const imageElement = (
    <AppImage
      src={image.url}
      alt={image.alt}
      height={101}
      width={180}
      style={{ width: "100%" }}
    />
  );

  return (
    <Card px={3} py={2} h="full" borderRadius="lg">
      <Flex direction="column" gap={4} justify="space-between">
        {/* Rendering Info Card image */}
        {(link && image && image.url) ? (
          <Link href={ur} title={heading} target={trg}>
            {imageElement}
          </Link>
        ) : (
          image && image.url && imageElement
        )}

        {/* Rendering Info Card Heading */}
        {(link && heading) ? (
          <Link href={ur} title={heading} target={trg}>
            {headingElement}
          </Link>
        ) : (
          heading && headingElement
        )}

        {/* Rendering Embed Code */}
        {embedcode && <div dangerouslySetInnerHTML={{ __html: embedcode }} />}

        {/* Rendering Info Card Description */}
        {description && <Text as="p">{description}</Text>}

        {/* Rendering Info card address */}
        {address && <Text as="address">{address}</Text>}

        {/* Rendering info card main address */}
        {address_main && (
          <Text fontSize="24" fontWeight="semibold">
            {address_main}
          </Text>
        )}

        {/* Rendering Info card district */}
        {district && <Text as="district">{district}</Text>}

        {/* Rendering Info card state */}
        {state && <Text as="state">{state}</Text>}

        {/* Rendering Contact Number */}
        {contact_no && (
          <Flex gap={5} align="center">
            <IconButton
              icon={<BsTelephone />}
              size="md"
              onClick={() => (window.location.href = `tel:${contact_no}`)}
              isRound
            />
            <Text as="span">{contact_no}</Text>
          </Flex>
        )}

        {/* Rendering Email */}
        {email && (
          <Flex gap={5} align="center">
            <Link href={`mailto:${email}`} target="_blank">
              <IconButton
                icon={<BsFillEnvelopeFill />}
                size="md"
                isRound
              />
            </Link>
            <Link href={`mailto:${email}`} target="_blank">
              <Text as="span">{email}</Text>
            </Link>
          </Flex>
        )}

        {/* Rendering Website */}
        {website && (
          <Flex gap={5} align="center">
            <Link href={`https://${website}`} target="_blank">
              <IconButton
                icon={<BsGlobe />}
                size="md"
                isRound
              />
            </Link>
            <Link href={`https://${website}`} target="_blank">
              <Text as="span">{website}</Text>
            </Link>
          </Flex>
        )}

        {/* Rendering Cities List */}
        {cities.length > 0 && (
          <Flex direction="column">
            {cities.map(({ _id, title, link, target = "_self" }) => (
              <Text color="orangePeel.900" key={_id}>
                <Link href={link} target={target}>
                  {title}
                </Link>
              </Text>
            ))}
          </Flex>
        )}

        {/* Rendering External Links */}
        {external_links.length > 0 && (
          <Flex
            direction="column"
            gap={2}
            sx={{
              a: {
                color: "orangePeel.900",
              },
            }}
          >
            {external_links.map(({ _id, text, ...restLinkProps }) => (
              <AppLink
                key={_id}
                {...restLinkProps}
                display="block"
                _hover={{
                  "svg path": {
                    fill: "orangePeel.900",
                  },
                }}
                animate
              >
                <Flex alignItems="center" justifyContent="space-between">
                  <Text as="span" maxW="sm">
                    {text}
                  </Text>
                  <Box>
                    <SvgArrow color="black" variant="large" />
                  </Box>
                </Flex>
              </AppLink>
            ))}
          </Flex>
        )}

        {/* Rendering Helpline Numbers */}
        {!isObjectEmpty(helpline_numbers) && (
          <Flex direction="column" gap={4}>
            {Object.keys(helpline_numbers).map((h) => (
              <Text key={h}>{helpline_numbers[h]}</Text>
            ))}
          </Flex>
        )}

        {/* Rendering Link Or Location Url */}
        {(link || location_url) && (
          <Flex justify="space-between" align="center">
            {link && (
              <Box display="inline-flex">
                <AppLink
                  title={link.title}
                  parent={link.parent}
                  slug={link.slug}
                  type={link.type}
                  variant="pill"
                  color="black"
                  arrow
                >
                  {link.text}
                </AppLink>
              </Box>
            )}
            {location_url && <AppLocationButton location_url={location_url} />}
          </Flex>
        )}
      </Flex>
    </Card>
  );
}
