"use client";

import dynamic from "next/dynamic";
import { string, shape, bool, arrayOf, oneOf } from "prop-types";
import { useMediaQuery } from "@chakra-ui/react";

import {
  COMPONENT_TYPES,
  BANNER_CATEGORIES,
} from "@/app/_constants/componentConstant";
import { isObjectEmpty } from "@/app/_utils";

import { BannerWrapper, ArticleBannerWrapper } from "./_wrappers";

const BannerImage = dynamic(() => import("./components/BannerImage"));
const BannerVideo = dynamic(() => import("./components/BannerVideo"));
const AdditionalInfo = dynamic(() => import("./components/AdditionalInfo"));
const MicroBannerInfo = dynamic(() => import("./components/MicroBannerInfo"));
const BannerCaption = dynamic(() => import("./components/BannerCaption"));
const AppPlayer = dynamic(() => import("@/app/_components/AppPlayer"));

const { BANNER } = COMPONENT_TYPES;
const {
  MAIN_BANNER,
  SUB_BANNER,
  ARTICLE_BANNER,
  ARTICLE_VIDEO_BANNER,
  MICRO_SITE_BANNER,
  VIDEO_BANNER,
} = BANNER_CATEGORIES;

export default function Banner({
  block,
  heading,
  description = "",
  caption = "",
  food_type = "",
  info = {},
  image = {},
  breadcrumb = {},
  gallery = {},
  video = {},
  video_bg = {},
  mediaQueryFallback,
}) {
  const { category } = block;
  const isSub = category === SUB_BANNER;
  const isMain = category === MAIN_BANNER || category === MICRO_SITE_BANNER || category === VIDEO_BANNER;
  const [isMobile] = useMediaQuery("(max-width: 767px)", {
    ssr: true,
    fallback: mediaQueryFallback,
  });

  const BannerWrapperElement = new Set([
    MAIN_BANNER,
    SUB_BANNER,
    VIDEO_BANNER,
    MICRO_SITE_BANNER,
  ]).has(category)
    ? BannerWrapper
    : ArticleBannerWrapper;

  return (
    <>
      <BannerWrapperElement
        isSub={category === SUB_BANNER}
        desc={description !== ""}
      >
        {/* Rendering Article Video Banner */}
        {!isObjectEmpty(video) && (
          <AppPlayer {...video} />
        )}

        {/* Rendering Banner Image */}
        {!isObjectEmpty(image) && (
          <BannerImage {...image} banner={isMain} isMobile={isMobile} />
        )}
        {/* Rendering Banner Background Video */}
        {!isObjectEmpty(video_bg) && (
          <BannerVideo heading={heading}
          description={description}
          video={video_bg}
          isMobile={isMobile} 
          banner={isMain}
          alt={''}
          mediaQueryFallback={mediaQueryFallback}
           />
        )}          
        {/* Rendering Banner Additional Info */}
        {!isObjectEmpty(breadcrumb) && heading && (
          <AdditionalInfo
            isSub={isSub}
            mediaQueryFallback={mediaQueryFallback}
            breadcrumb={breadcrumb}
            heading={heading}
            description={description}
          />
        )}

        {/* Rendering Micro Site Banner */}
        {!isObjectEmpty(info) && <MicroBannerInfo {...info} />}

        {/* Rendering Banner Caption With Gallery Icon */}
        {(caption || !isObjectEmpty(gallery)) && (
          <BannerCaption
            caption={caption}
            gallery={gallery}
            food_type={food_type}
            isMain={isMain}
            isMobile={isMobile}
          />
        )}
      </BannerWrapperElement>
    </>
  );
}

/**
 * PropTypes for the Application
 * Banner component
 */
Banner.propTypes = {
  block: shape({
    type: oneOf([BANNER]).isRequired,
    category: oneOf([
      MAIN_BANNER,
      SUB_BANNER,
      ARTICLE_BANNER,
      ARTICLE_VIDEO_BANNER,
      MICRO_SITE_BANNER,
    ]).isRequired,
  }).isRequired,
  heading: string,
  description: string,
  caption: string,
  food_type: string,
  image: shape({
    url: shape({
      desktop: string,
      mobile: string,
    }),
    alt: string,
  }),
  video: shape({
    type: oneOf(["youtube", "static"]),
    url: string,
    poster_image: string,
  }),
  breadcrumb: arrayOf(
    shape({
      _id: string,
      title: string,
      isCurrent: bool,
      link: shape({
        parent: string,
        slug: string,
      }),
    })
  ),
};
