import { motion } from "framer-motion";
import { string, oneOf, node } from "prop-types";
import { useState, useEffect } from "react";

import FilePlayer from "react-player/file";
import YouTubePlayer from "react-player/youtube";

import { Box, Skeleton } from "@chakra-ui/react";
import { AppImage } from "@/app/_components";
import { ReactSVG } from "react-svg";

export default function AppPlayer({ type, url, poster_image = "", children }) {
  const [hasWindow, setHasWindow] = useState(false);

  // Checking the Window is Defined Or Not
  // While in the Client Mode
  useEffect(() => {
    if (typeof window !== "undefined") setHasWindow(true);
  }, []);

  // Player Light Image
  const playerLightImage =
    poster_image !== "" ? (
      <AppImage
        src={poster_image}
        fill
        sizes="75vw"
        style={{
          height: "100%",
          width: "100%",
          objectFit: "cover",
          borderRadius: "0.375rem",
        }}
        alt="youtube"
      />
    ) : null;

  // Video Player common props
  const playerProps = {
    className: "video-player",
    url,
    width: "100%",
    height: "100%",
    fallback: <Skeleton h="full" />,
    ...(poster_image !== "" && {
      light: playerLightImage,
    }),
    playIcon: (
      <svg
        className="light-icon"
        fill="none"
        viewBox="0 0 20 20"
        xmlns="http://www.w3.org/2000/svg"
      >
        <motion.g>
          <motion.circle
            cx="10"
            cy="10"
            r="8"
            strokeWidth="0.5"
            stroke="#fff"
            animate={{
              opacity: [0, 1, 1, 1, 1],
              strokeWidth: [0.2, 0.8, 0.8, 0.8, 0.8],
            }}
            transition={{
              duration: 2,
              repeat: Infinity,
              repeatType: "loop",
              repeatDelay: 3,
            }}
          />
          <motion.polygon
            animate={{
              opacity: [0, 0.5, 0.3, 0.5, 1],
            }}
            transition={{
              duration: 2,
              repeat: Infinity,
              repeatType: "loop",
              repeatDelay: 3,
            }}
            points="13.5 6 8 6 10.75 0.5 13.5 6"
            fill="#fff"
          />
        </motion.g>
      </svg>
    ),
    controls: type === "static",
  };

  return (
    <Skeleton isLoaded={hasWindow}>
      <Box
        pos="relative"
        pt="56.25%"
        sx={{
          ".video-player": {
            pos: "absolute",
            top: 0,
            left: 0,
            " > .react-player__preview": {
              position: "relative",
            },
          },
          ".light-icon": {
            polygon: {
              transform: "translate(14px, -1px) rotate(90deg)",
            },
            zIndex: 99,
            w: {
              base: "50px",
              md: "100px",
              lg: "150px",
            },
            h: {
              base: "50px",
              md: "100px",
              lg: "150px",
            },
          },
        }}
      >
        {hasWindow && (
          <>
            {type === "static" ? (
              <FilePlayer {...playerProps} />
            ) : (
              <YouTubePlayer
                {...playerProps}
                config={{
                  youtube: { playerVars: { disablekb: "1", controls: "0" } },
                  embedOptions: { iv_load_policy: 3 },
                }}
              />
            )}
          </>
        )}
        {children}
      </Box>
    </Skeleton>
  );
}

AppPlayer.propTypes = {
  type: oneOf(["youtube", "static"]).isRequired,
  url: string.isRequired,
  poster_image: string,
  children: node,
};
