"use client";

import { bool, node, oneOf } from "prop-types";
import { Box, Flex, Grid, useMediaQuery } from "@chakra-ui/react";

import useEmblaCarousel from "embla-carousel-react";
import { DotButton, useDotButton } from "./EmblaCarouselDotButton";
import { SliderButton, usePrevNextButtons } from "./EmblaCarouselArrowButton";

import { pxToRem } from "@/app/_utils";

// Container component for Embla
export const EmblaContainer = ({ type = "flex", children, ...restProps }) => {
  if (type === "flex") {
    return <Flex {...restProps}>{children}</Flex>;
  }

  return <Grid {...restProps}>{children}</Grid>;
};

EmblaContainer.propTypes = {
  type: oneOf(["flex", "grid"]),
  children: node.isRequired,
};

function AppCarousel({
  showDots = false,
  showArrows = false,
  overflow = true,
  arrowPosition = "right",
  arrowColor = "white",
  dotPosition = "top",
  slideToScroll = 1,
  hasBottomLink = false,
  arrowBottomPadding = true,
  children,
  items
}) {
  const [isMobile] = useMediaQuery(["(max-width: 991px)"]);

  const [emblaRef, emblaApi] = useEmblaCarousel({
    align: "center",
    slidesToScroll: slideToScroll,
    ...(slideToScroll > 1 && {
      breakpoints: {
        "(max-width: 767px)": {
          slidesToScroll: 1,
        },
        "(min-width: 768px) and (max-width: 991px)": {
          slidesToScroll: 2,
        },
        "(min-width: 992px) and (max-width: 1535px)": {
          slidesToScroll: 3,
        },
      },
    }),
  });

  const rightArrowStyle = {
    bottom: hasBottomLink ? "100px" : "40px",
    right: "40px",
  };

  const centerArrowStyle = {
    right: "0",
    left: "0",
    bottom: "0",
    display: "flex",
    justifyContent: "center",
  };

  const VideoBGCenterArrowStyle = {
    right: "0",
    left: "0",
    bottom: "150px", // Adjust this value to move the arrows below the AnimationWrapper
    display: "flex",
    justifyContent: "center",
    position: "absolute", // Ensure the arrows are positioned absolutely within the parent
  };

  const leftArrowStyle = {
    left: "0",
  };

  const _arrowBoxStyle = {
    pos: {
      base: "relative",
      lg: "absolute",
    },
  };

  const dotBoxTopStyle = {
    top: "0",
  };

  const dotBoxBottomStyle = {
    bottom: "0",
  };

  const _dotBoxStyle = {
    pos: "absolute",
    left: "0",
    right: "0",
    margin: "0 auto",
    textAlign: "center",

    ...(dotPosition === "top" && dotBoxTopStyle),
    ...(dotPosition === "bottom" && dotBoxBottomStyle),
  };

  // Embla Functions for handling
  // Carousel dots and Nav Btns
  const { selectedIndex, scrollSnaps, onDotButtonClick } =
    useDotButton(emblaApi);

  const {
    prevBtnDisabled,
    nextBtnDisabled,
    onPrevButtonClick,
    onNextButtonClick,
  } = usePrevNextButtons(emblaApi);

  // Change pt in array format
  // it can be used in responsive views
  const _appCarouselProps = {
    pos: "relative",
    pt: showDots ? pxToRem(50) : 0,
    pb: {
      base: 0,
      lg: arrowPosition === "VideoBGCenter" || arrowPosition === "center" || hasBottomLink ? pxToRem(80) : 0,
    },
    ...(overflow && {
      overflow: "hidden",
    }),
    ...(!overflow && {
      _before: {
        bottom: 0,
        content: '""',
        display: "block",
        pos: "absolute",
        top: 0,
        zIndex: "999",
        backgroundColor: "rgb(255 255 255 / 65%)",
        left: 0,
        transform: "translateX(-100%)",
        w: "300%",
      },
      _after: {
        bottom: 0,
        content: '""',
        display: "block",
        pos: "absolute",
        top: 0,
        zIndex: "999",
        backgroundColor: "rgb(255 255 255 / 65%)",
        right: "0",
        transform: "translateX(100%)",
        w: "150%",
      },
    }),
  };
  let cn=1;
  isMobile ? cn=1 : cn=4;
  return (
    <Box ref={emblaRef} {..._appCarouselProps}>
      {children}

      {/* Carousel dot component */}
      {showDots && (
        <Box {..._dotBoxStyle}>
          <Box
            className="carousel__dots"
            sx={{
              "button.carousel__dot": {
                width: "7px",
                height: "7px",
                backgroundColor: "gray.400",
                borderRadius: "full",
                margin: "0 3px",
                touchAction: "manipulation",
                transition: "all 0.3s linear",
              },
              "button.carousel__dot--selected": {
                height: "7px",
                width: "24px",
                backgroundColor: "orangePeel.900",
              },
            }}
          >
            {scrollSnaps.map((_, index) => (
              <DotButton
                key={index}
                onClick={() => onDotButtonClick(index)}
                className={"carousel__dot".concat(
                  index === selectedIndex ? " carousel__dot--selected" : ""
                )}
              />
            ))}
          </Box>
        </Box>
      )}

      {/* Carousel Arrow Btn */}
      {showArrows && (
        <Box {..._arrowBoxStyle} sx={{
          "@media screen and (min-width: 992px)": {
            ...(arrowPosition === "right" && rightArrowStyle),
            ...(arrowPosition === "center" && centerArrowStyle),
            ...(arrowPosition === "VideoBGCenter" && VideoBGCenterArrowStyle),
            ...(arrowPosition === "left" && leftArrowStyle),
          },
          "@media screen and (max-width: 991px)": {
            top: "auto",
            bottom: 0,
            insetX: 0,
            width: "full",
            px: 0,
            pt: 8,
            ...(arrowBottomPadding && {
              pb: 8,
            }),
          },
        }}>
         {/*  {items && items.length > cn && ( */}
          {items && (
            <Box
            display="flex"
            justifyContent={{ base: "center", lg: "flex-start" }}
            gap={3}
          >
            {!prevBtnDisabled && (
              <SliderButton
                direction="rtl"
                onClick={onPrevButtonClick}
                disabled={prevBtnDisabled}
                color={arrowColor}
              />
            )}
            {!nextBtnDisabled && (
              <SliderButton
                direction="ltr"
                onClick={onNextButtonClick}
                disabled={nextBtnDisabled}
                color={arrowColor}
              />
            )}
          </Box>

          )}
          
        </Box>
      )}
    </Box>
  );
}

AppCarousel.propTypes = {
  showDots: bool,
  showArrows: bool,
  overflow: bool,
  hasBottomLink: bool,
  dotPosition: oneOf(["top", "bottom"]),
  arrowColor: oneOf(["white", "black"]),
  arrowPosition: oneOf(["right", "left", "center"]),
  children: node.isRequired,
};

export default AppCarousel;
